import bpy, bmesh, mathutils
import time, struct, os
from bpy.props import *

import glob

import nrfile
import nrtools


######################################
# Blender versions workaround
######################################
def ver_blender():
    if bpy.app.version < (2, 80, 0):
        return 0
    return 1


g_logger = None
def setLogger(logger):
    global g_logger
    g_logger = logger

def writeLog(s):
    global g_logger
    if g_logger:
        g_logger.write(s)


def isImageLoaded(img):
    if (0==img.size[0]) and (img.size[1]==0):
        return False
    return True


class GroupManager(object):
    def __init__(self):
        self.colDict = {}

    def getGroup(self, fileName):

        if 0 == ver_blender():
            # Blender < 2.8 has no collections??????
            return None

        baseFileName = os.path.basename(fileName)
        arr = baseFileName.split("_")

        meshgrpName = None
        if len(arr) >= 2:
            meshgrpName = "grp_" + arr[1]

        grp = None
        if meshgrpName:
            # Name parsed successfully
            grp = self.colDict.get(meshgrpName, "notfound")
            if grp == "notfound":
                # Create group
                grp = None
                grp = bpy.data.collections.new(name=meshgrpName)  # Create new collection
                bpy.context.scene.collection.children.link(grp)   # Add to scene
                self.colDict[meshgrpName] = grp
        return grp


class MaterialCache(object):    
    def __init__(self):
        self.__matId = 0
        self.__materialCache = {}
        self.__textureCache  = {}
        self.loadedImgs = []
        self.failedImgs = []

    def __matHash(self, texList):
        res = ""
        for texFile in texList:
            res = res + texFile
        return res


    def createMaterial(self, texList):
        if 0 == len(texList):
            return None

        matHash = self.__matHash(texList)

        mat = self.__materialCache.get(matHash, "materialnotfound")
        if "materialnotfound" == mat:
            mat = None

            # Create new material
            matName = "mat_" + str(self.__matId)
            self.__matId = self.__matId + 1
            if 0 == ver_blender():
                # Blender < 2.8
                mat = self.__createMat27(matName, texList)
            else:
                # Blender >= 2.8
                mat = self.__createMat28(matName, texList)


            # Add to cache.
            # If mat == None then material create error
            self.__materialCache[matHash] = mat
        return mat


    # Blender < 2.8
    def __createImageTexture(self, fullpath):
        tex = self.__textureCache.get(fullpath, "notexturefound")
        if "notexturefound" == tex:
            tex = None
            #writeLog("--> Image load: {:s}".format(fullpath))
            try:
                img = bpy.data.images.load(fullpath, True)
                if isImageLoaded(img):
                    writeLog("--> SUCCESSFULL load: {:s}".format(fullpath))
                    self.loadedImgs.append(fullpath)
                    # Image loaded successfully
                    #   Create texture
                    tex = bpy.data.textures.new(fullpath, type='IMAGE')
                    tex.image = img
                else:
                    writeLog("--> FAILED to load: {:s}".format(fullpath))
                    self.failedImgs.append(fullpath)
                    bpy.data.images.remove(img)
                    tex = None
                    img = None
            except Exception as e:
                tex = None

            self.__textureCache[fullpath] = tex
        return tex


    def __createMat27(self, matName, texList):
        texObjList = []
        for texFile in texList:
            tex = self.__createImageTexture(texFile)
            if tex:
                texObjList.append(tex)

        # Material textures load failed.
        # No textures -> exit
        if 0 == len(texObjList):
            return None

        # We have at least one texture -> create material
        mat = bpy.data.materials.new(matName)
        first = True
        i = 0
        for tex in texObjList:
            slot = mat.texture_slots.create(i)
            slot.texture = tex
            slot.use = first
            first = False
            i = i + 1
        return mat


    # Blender >= 2.8
    def __createImage(self, fullpath):
        img = self.__textureCache.get(fullpath, "notexturefound")
        if "notexturefound" == img:
            img = None
            #writeLog("--> Image load: {:s}".format(fullpath))
            try:
                img = bpy.data.images.load(fullpath)
                if isImageLoaded(img):
                    writeLog("--> SUCCESSFULL load: {:s}".format(fullpath))
                    self.loadedImgs.append(fullpath)
                    self.__textureCache[fullpath] = img
                else:
                    writeLog("--> FAILED to load: {:s}".format(fullpath))
                    self.failedImgs.append(fullpath)
                    bpy.data.images.remove(img)
                    img = None
            except Exception as e:
                img = None

            self.__textureCache[fullpath] = img

        return img


    def __createMat28(self, matName, texList):
        # Blender >= 2.8
        imgObjList = []
        for texFile in texList:
            img = self.__createImage(texFile)
            if img:
                imgObjList.append(img)

        if 0 == len(imgObjList):
            return None

        mat = bpy.data.materials.new(matName)
        mat.use_nodes = True
        ## CF2 user fix
        #for i, ib in enumerate(imgObjList):
        #    b = int(str(i/3)[:1])+1
        #    texImage = mat.node_tree.nodes.new('ShaderNodeTexImage')
        #    texImage.image = imgObjList[i]
        #    texImage.location.x -= b*300
        #    texImage.location.y -= (i-b*3)*300+600

        #img = imgObjList[0]
        #while True:
        #    bsdf = mat.node_tree.nodes["Principled BSDF"]  #   Diffuse BSDF?
        #    texImage = mat.node_tree.nodes.new('ShaderNodeTexImage')
        #    texImage.image = img
        #    mat.node_tree.links.new(bsdf.inputs['Base Color'], texImage.outputs['Color'])
        #    break

        baseColorConnected = False
        bsdf = mat.node_tree.nodes["Principled BSDF"]  #   Diffuse BSDF?
        for i, ib in enumerate(imgObjList):
            b = int(str(i/3)[:1])+1
            texImage = mat.node_tree.nodes.new('ShaderNodeTexImage')
            texImage.image = imgObjList[i]
            texImage.location.x -= b*300
            texImage.location.y -= (i-b*3)*300+600
            if baseColorConnected == False:
                mat.node_tree.links.new(bsdf.inputs['Base Color'], texImage.outputs['Color'])
                baseColorConnected = True

        return mat


def createTexCoordsScatter(bm, options, vert, vertexData, vatrs):
    texScatterU  = options['texScatterU']  # [attrIdx, compIdx]
    texScatterV  = options['texScatterV']  # [attrIdx, compIdx]

    texCoords = nrtools.unpackVertexComponentVaAsList(vert, vertexData, vatrs, [texScatterU, texScatterV])
    if None == texCoords:
        return

    uv_lay = bm.loops.layers.uv.new('uv')

    for face in bm.faces:
        for vv in face.loops:
            vertIndex = vv.vert.index
            uv = mathutils.Vector(texCoords[vertIndex])
            vv[uv_lay].uv.x = uv.x
            vv[uv_lay].uv.y = 1.0 - uv.y


def createTexCoords(bm, options, vert, vertexData, vatrs):
    # Create UV maps
    texCoordAttribName  = options['texCoordAttribName']  # Default TEXCOORD
    texUvComponentU_Idx = options['texUvComponentIdx'][0]
    texUvComponentV_Idx = options['texUvComponentIdx'][1]

    texcoordVaList = vatrs.findSemantic(texCoordAttribName)
    if len(texcoordVaList):
        # UV coordinates loop
        #TEXCOORD_0
        #TEXCOORD_1
        #TEXCOORD_2
        #TEXCOORD_3
        #TEXCOORD_4
        for uvIdx in range(0, len(texcoordVaList)):
            tcVa = texcoordVaList[uvIdx]   # Vertex attribute for texture coordinates

            #v0 uvw
            #v1 uvw
            #v2 uvw
            #....
            layerTextureCoordinates = nrtools.unpackVertexComponentAsList(vert, vertexData, tcVa)
            layerTextureCoordinates = nrtools.extractComponentsAsList(layerTextureCoordinates, [texUvComponentU_Idx, texUvComponentV_Idx])
            if 0 == len(layerTextureCoordinates):
                writeLog("No texture coordinates")
                continue
            if len(layerTextureCoordinates[0]) < 2:
                writeLog("UV components < 2")
                continue

            uv_lay = bm.loops.layers.uv.new('uv_' + str(uvIdx))

            for face in bm.faces:
                for vv in face.loops:
                    vertIndex = vv.vert.index
                    uv = mathutils.Vector(layerTextureCoordinates[vertIndex])
                    vv[uv_lay].uv.x = uv.x
                    vv[uv_lay].uv.y = 1.0 - uv.y
    else:
        writeLog("Texture coords attribute {:s} not found".format(texCoordAttribName))


def setFarClipDistance():
    # Set far clip distance
    for a in bpy.context.screen.areas:
        if a.type == 'VIEW_3D':
            for s in a.spaces:
                if s.type == 'VIEW_3D':
                    s.clip_end = 1000000.0
